<?php
/**
 * Free License Manager - E-Mail Verifizierung für kostenlose Version
 */

if (!defined('ABSPATH')) {
    exit;
}

class GermanFence_Free_License {
    
    private $table_name;
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'germanfence_free_users';
        
        // AJAX Handler
        add_action('wp_ajax_germanfence_register_free', array($this, 'ajax_register_free'));
        add_action('wp_ajax_nopriv_germanfence_register_free', array($this, 'ajax_register_free'));
        add_action('wp_ajax_germanfence_activate_free_key', array($this, 'ajax_activate_free_key'));
        add_action('wp_ajax_germanfence_deactivate_free', array($this, 'ajax_deactivate_free'));
        
        // Verification Handler
        add_action('admin_init', array($this, 'handle_verification'));
    }
    
    /**
     * Registriert einen neuen Free-User
     */
    public function register_email($email) {
        global $wpdb;
        
        // E-Mail validieren
        if (!is_email($email)) {
            return array('success' => false, 'message' => 'Ungültige E-Mail-Adresse');
        }
        
        // Prüfen ob E-Mail bereits registriert
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE email = %s",
            $email
        ));
        
        if ($existing) {
            if ($existing->is_verified) {
                return array('success' => false, 'message' => 'Diese E-Mail ist bereits verifiziert');
            } else {
                // Bestätigungsmail erneut senden
                $this->send_verification_email($email, $existing->verification_token);
                return array('success' => true, 'message' => 'Bestätigungsmail wurde erneut gesendet!');
            }
        }
        
        // Neuen Token generieren
        $token = bin2hex(random_bytes(32));
        
        // In Datenbank speichern
        $result = $wpdb->insert(
            $this->table_name,
            array(
                'email' => $email,
                'verification_token' => $token,
                'is_verified' => 0,
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%d', '%s')
        );
        
        if (!$result) {
            return array('success' => false, 'message' => 'Fehler beim Speichern');
        }
        
        // Bestätigungsmail senden
        $sent = $this->send_verification_email($email, $token);
        
        if ($sent) {
            return array('success' => true, 'message' => 'Bestätigungsmail wurde gesendet! Bitte prüfe dein Postfach.');
        } else {
            return array('success' => false, 'message' => 'Fehler beim Versenden der E-Mail');
        }
    }
    
    /**
     * Sendet Verifizierungs-E-Mail
     */
    private function send_verification_email($email, $token) {
        $verification_url = admin_url('admin.php?page=germanfence&tab=license&verify_token=' . $token);
        $portal_url = 'https://portal.germanfence.de/login';
        $password_reset_url = 'https://portal.germanfence.de/register?email=' . urlencode($email) . '&setPassword=true';
        
        $subject = '🛡️ GermanFence - E-Mail bestätigen';
        
        $message = "Hallo!\n\n";
        $message .= "Danke für deine Registrierung bei GermanFence!\n\n";
        $message .= "Bitte bestätige deine E-Mail-Adresse, um die kostenlose Version zu aktivieren:\n\n";
        $message .= "👉 " . $verification_url . "\n\n";
        $message .= "Der Link ist 24 Stunden gültig.\n\n";
        $message .= "══════════════════════════════════════════\n\n";
        $message .= "📦 NACH DER VERIFIZIERUNG:\n\n";
        $message .= "• Du erhältst automatisch einen FREE-LICENSE-KEY\n";
        $message .= "• Damit kannst du GermanFence auf weiteren Domains aktivieren\n";
        $message .= "• Der Key wird dir nach der Verifizierung im Plugin angezeigt\n\n";
        $message .= "══════════════════════════════════════════\n\n";
        $message .= "🌐 PORTAL-ZUGANG:\n\n";
        $message .= "Ab jetzt kannst du dich im GermanFence Portal einloggen:\n";
        $message .= "👉 " . $portal_url . "\n\n";
        $message .= "Falls du noch kein Passwort gesetzt hast, kannst du es hier erstellen:\n";
        $message .= "👉 " . $password_reset_url . "\n\n";
        $message .= "══════════════════════════════════════════\n\n";
        $message .= "Viel Erfolg mit GermanFence!\n";
        $message .= "Dein GermanFence Team 🇩🇪\n\n";
        $message .= "Website: https://germanfence.de\n";
        $message .= "Support: support@germanfence.de";
        
        $headers = array('Content-Type: text/plain; charset=UTF-8');
        
        return wp_mail($email, $subject, $message, $headers);
    }
    
    /**
     * Verifiziert einen Token
     */
    public function verify_token($token) {
        global $wpdb;
        
        $user = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE verification_token = %s",
            $token
        ));
        
        if (!$user) {
            return array('success' => false, 'message' => 'Ungültiger Verifizierungslink');
        }
        
        if ($user->is_verified) {
            return array('success' => false, 'message' => 'Diese E-Mail wurde bereits verifiziert');
        }
        
        // Prüfe ob Token älter als 24 Stunden
        $created = strtotime($user->created_at);
        $now = current_time('timestamp');
        
        if (($now - $created) > 86400) {
            return array('success' => false, 'message' => 'Der Verifizierungslink ist abgelaufen (24 Stunden)');
        }
        
        // Free-License-Key generieren
        $license_key = 'GS-FREE-' . strtoupper(substr(bin2hex(random_bytes(8)), 0, 12));
        
        // Verifizieren
        $result = $wpdb->update(
            $this->table_name,
            array(
                'is_verified' => 1,
                'verified_at' => current_time('mysql'),
                'license_key' => $license_key
            ),
            array('id' => $user->id),
            array('%d', '%s', '%s'),
            array('%d')
        );
        
        if ($result !== false) {
            // Speichere verifizierte E-Mail in Options (als String '1' für WordPress-Kompatibilität)
            update_option('germanfence_free_email', $user->email);
            update_option('germanfence_free_verified', '1');
            update_option('germanfence_free_license_key', $license_key);
            
            GermanFence_Logger::log('[FREE-LICENSE] E-Mail verifiziert: ' . $user->email . ' | Key: ' . $license_key);
            GermanFence_Logger::log('[FREE-LICENSE] Option gespeichert: ' . get_option('germanfence_free_verified'));
            
            return array(
                'success' => true, 
                'message' => 'E-Mail erfolgreich verifiziert! Du kannst jetzt die kostenlose Version nutzen.',
                'license_key' => $license_key
            );
        }
        
        return array('success' => false, 'message' => 'Fehler bei der Verifizierung');
    }
    
    /**
     * Prüft ob Free-Lizenz aktiv ist
     */
    public function is_free_active() {
        $verified = get_option('germanfence_free_verified', false);
        
        // WordPress speichert manchmal als String '1' oder als Boolean true
        $is_active = ($verified === true || $verified === '1' || $verified === 1);
        
        GermanFence_Logger::log('[FREE-LICENSE] Prüfe Status: ' . ($is_active ? 'AKTIV' : 'INAKTIV') . ' (Wert: ' . var_export($verified, true) . ')');
        
        return $is_active;
    }
    
    /**
     * Gibt verifizierte E-Mail zurück
     */
    public function get_verified_email() {
        return get_option('germanfence_free_email', '');
    }
    
    /**
     * AJAX Handler für Registrierung
     */
    public function ajax_register_free() {
        check_ajax_referer('germanfence_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Keine Berechtigung');
        }
        
        $email = sanitize_email($_POST['email'] ?? '');
        
        $result = $this->register_email($email);
        
        if ($result['success']) {
            wp_send_json_success($result['message']);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX Handler für Key-Aktivierung
     */
    public function ajax_activate_free_key() {
        check_ajax_referer('germanfence_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Keine Berechtigung');
        }
        
        $key = sanitize_text_field($_POST['license_key'] ?? '');
        
        $result = $this->activate_with_key($key);
        
        if ($result['success']) {
            wp_send_json_success($result['message']);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX Handler für Deaktivierung
     */
    public function ajax_deactivate_free() {
        check_ajax_referer('germanfence_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Keine Berechtigung');
        }
        
        $result = $this->deactivate_free();
        
        if ($result['success']) {
            wp_send_json_success($result['message']);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * Handle Verification in Admin
     */
    public function handle_verification() {
        if (isset($_GET['page']) && $_GET['page'] === 'germanfence' && isset($_GET['verify_token'])) {
            $token = sanitize_text_field($_GET['verify_token']);
            $result = $this->verify_token($token);
            
            if ($result['success']) {
                // WICHTIG: Cache leeren damit neue Werte sofort verfügbar sind
                wp_cache_delete('germanfence_free_verified', 'options');
                wp_cache_delete('germanfence_free_email', 'options');
                
                GermanFence_Logger::log('[FREE-LICENSE] Verifizierung erfolgreich, Cache geleert, Redirect zu License-Tab');
                
                // Redirect nach Verifizierung um Token aus URL zu entfernen - Toast-Meldung über JS
                wp_redirect(admin_url('admin.php?page=germanfence&tab=license&verified=1'));
                exit;
            } else {
                // Fehler über URL-Parameter für Toast-Meldung
                wp_redirect(admin_url('admin.php?page=germanfence&tab=license&verify_error=' . urlencode($result['message'])));
                exit;
            }
        }
        
        // Toast-Meldungen werden jetzt über JavaScript gehandhabt (admin.js)
        // Keine WP admin_notices mehr verwenden!
    }
    
    /**
     * Aktiviert Lizenz mit Key (FREE, PRO, manuell generiert - alle Formate)
     */
    public function activate_with_key($key) {
        global $wpdb;
        
        // Key validieren und normalisieren
        $key = strtoupper(trim(sanitize_text_field($key)));
        
        if (empty($key) || strlen($key) < 8) {
            return array('success' => false, 'message' => 'Ungültiger License-Key (zu kurz)');
        }
        
        // Akzeptiere ALLE Key-Formate:
        // - GS-FREE-XXXX (Free-Keys)
        // - GS-PRO-XXXX, GS-SINGLE-XXXX, GS-FREELANCER-XXXX, GS-AGENCY-XXXX (Pro-Keys)
        // - Beliebige andere alphanumerische Keys mit Bindestrichen
        if (!preg_match('/^[A-Z0-9][A-Z0-9\-]{6,}[A-Z0-9]$/', $key)) {
            return array('success' => false, 'message' => 'Ungültiges Key-Format. Nur Buchstaben, Zahlen und Bindestriche erlaubt.');
        }
        
        // Erst lokal prüfen ob Key existiert (für FREE-Keys)
        $user = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE license_key = %s AND is_verified = 1",
            $key
        ));
        
        // Wenn lokal gefunden, E-Mail übernehmen
        // Wenn NICHT lokal gefunden, versuche von API zu holen (PRO-Keys)
        $email = '';
        if ($user) {
            $email = $user->email;
        }
        
        // Key-Typ bestimmen
        $key_type = 'CUSTOM';
        if (strpos($key, 'GS-FREE-') === 0) {
            $key_type = 'FREE';
        } elseif (strpos($key, 'GS-PRO-') === 0 || strpos($key, 'GS-SINGLE-') === 0 || 
                  strpos($key, 'GS-FREELANCER-') === 0 || strpos($key, 'GS-AGENCY-') === 0) {
            $key_type = 'PRO';
        }
        
        // Key aktivieren
        update_option('germanfence_free_email', $email);
        update_option('germanfence_free_verified', '1');
        update_option('germanfence_free_license_key', $key);
        
        // Bei PRO-Keys auch in die Premium-Settings speichern
        if ($key_type === 'PRO') {
            $settings = get_option('germanfence_settings', array());
            $settings['license_key'] = $key;
            update_option('germanfence_settings', $settings);
            
            // License-Cache leeren für Revalidierung
            delete_transient('germanfence_license_data');
            
            // SOFORT API-Validierung durchführen um Features zu laden
            require_once GERMANFENCE_PLUGIN_DIR . 'includes/class-license.php';
            $license = new GermanFence_License();
            $validation = $license->validate_license($key);
            
            // Email aus API-Response holen wenn nicht lokal gefunden
            if (empty($email) && isset($validation['license']['userEmail'])) {
                $email = $validation['license']['userEmail'];
            }
            
            GermanFence_Logger::log('[LICENSE] API-Validierung nach Aktivierung: ' . json_encode($validation));
        }
        
        // Cache leeren
        wp_cache_delete('germanfence_free_verified', 'options');
        wp_cache_delete('germanfence_free_email', 'options');
        wp_cache_delete('germanfence_free_license_key', 'options');
        
        GermanFence_Logger::log('[LICENSE] Mit Key aktiviert: ' . $key . ' | Typ: ' . $key_type . ' | E-Mail: ' . $email);
        
        // Erfolgs-Nachricht mit Lizenztyp
        $type_names = array(
            'FREE' => 'FREE',
            'PRO' => 'PRO',
            'SINGLE' => 'SINGLE',
            'FREELANCER' => 'FREELANCER',
            'AGENCY' => 'AGENCY',
            'CUSTOM' => 'CUSTOM'
        );
        
        // Detaillierter Typ für spezielle Keys
        $detailed_type = $key_type;
        if (strpos($key, 'GS-SINGLE-') === 0) $detailed_type = 'SINGLE';
        elseif (strpos($key, 'GS-FREELANCER-') === 0) $detailed_type = 'FREELANCER';
        elseif (strpos($key, 'GS-AGENCY-') === 0) $detailed_type = 'AGENCY';
        
        $type_display = isset($type_names[$detailed_type]) ? $type_names[$detailed_type] : $detailed_type;
        $success_msg = '✅ ' . $type_display . '-Lizenz erfolgreich aktiviert!';
        
        return array('success' => true, 'message' => $success_msg);
    }
    
    /**
     * Gibt aktuellen License-Key zurück
     */
    public function get_license_key() {
        return get_option('germanfence_free_license_key', '');
    }
    
    /**
     * Deaktiviert Free-Lizenz
     */
    public function deactivate_free() {
        delete_option('germanfence_free_email');
        delete_option('germanfence_free_verified');
        delete_option('germanfence_free_license_key');
        
        // Cache leeren
        wp_cache_delete('germanfence_free_verified', 'options');
        wp_cache_delete('germanfence_free_email', 'options');
        wp_cache_delete('germanfence_free_license_key', 'options');
        
        GermanFence_Logger::log('[FREE-LICENSE] Kostenlose Version deaktiviert');
        
        return array('success' => true, 'message' => 'Kostenlose Version deaktiviert');
    }
    
    /**
     * Debug-Info für Free-Lizenz
     */
    public function get_debug_info() {
        global $wpdb;
        
        $verified_option = get_option('germanfence_free_verified', 'NICHT GESETZT');
        $email_option = get_option('germanfence_free_email', 'NICHT GESETZT');
        
        $users = $wpdb->get_results("SELECT * FROM {$this->table_name}");
        
        return array(
            'option_verified' => $verified_option,
            'option_email' => $email_option,
            'is_active' => $this->is_free_active(),
            'users_in_db' => $users
        );
    }
}

